"use client";
import { FormEvent, useEffect, useMemo, useRef, useState } from "react";
import { useSession } from "next-auth/react";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";
import { useRouter } from "next/router";
import {
  ArrowLeft,
  Save,
  Mail,
  User as UserIcon,
  AlertCircle,
  CheckCircle2,
  Copy,
  ExternalLink,
  Upload,
  X,
  Image as ImageIcon,
} from "lucide-react";
import { Dialog } from "@/components/Dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

type CreateResp =
  | {
      ok: true;
      txHash: string;
      jsonHash: string;
      contract: string;
      ticket: any;
    }
  | { ok: false; error: string };

type Department = { id: string; name: string };
type LocationItem = { id: string; name: string; departmentId: string };
type ElementItem = { id: string; name: string; locationId: string };

export default function NewTicketPage() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const { t } = useLanguage();
  const {
    department,
    departmentId: queryDepartmentId,
    locationId: queryLocationId,
    elementId: queryElementId,
    title: queryTitle,
    description: queryDescription,
    taskId,
  } = router.query;

  const [title, setTitle] = useState("");
  const [description, setDescription] = useState("");
  const [severity, setSeverity] = useState(2);
  const [uploadedFiles, setUploadedFiles] = useState<File[]>([]);

  const [anonymousEmail, setAnonymousEmail] = useState("");
  const [anonymousName, setAnonymousName] = useState("");

  const [deps, setDeps] = useState<Department[]>([]);
  const [depId, setDepId] = useState<string>("");
  const [locations, setLocations] = useState<LocationItem[]>([]);
  const [locationId, setLocationId] = useState<string>("");
  const [els, setEls] = useState<ElementItem[]>([]);
  const [elId, setElId] = useState<string>("");

  const [loading, setLoading] = useState(false);
  const [result, setResult] = useState<CreateResp | null>(null);
  const [err, setErr] = useState<string | null>(null);

  // dialog state
  const [dialogOpen, setDialogOpen] = useState(false);

  const legacyDepartment = useMemo(
    () => deps.find((d) => d.id === depId)?.name ?? "",
    [deps, depId]
  );

  useEffect(() => {
    (async () => {
      try {
        const r = await fetch("/api/departments");
        const j = await r.json();
        if (j.ok) {
          setDeps(j.items || []);
          // Priority: queryDepartmentId > department > current depId > first item
          if (queryDepartmentId && typeof queryDepartmentId === "string") {
            const foundDept = j.items.find(
              (d: Department) => d.id === queryDepartmentId
            );
            if (foundDept) {
              setDepId(foundDept.id);
            } else if (!depId && j.items?.length) {
              setDepId(j.items[0].id);
            }
          } else if (department && typeof department === "string") {
            const foundDept = j.items.find(
              (d: Department) => d.id === department
            );
            if (foundDept) {
              setDepId(foundDept.id);
            } else if (!depId && j.items?.length) {
              setDepId(j.items[0].id);
            }
          } else if (!depId && j.items?.length) {
            setDepId(j.items[0].id);
          }
        } else {
          setErr(j.error || "Failed to load departments");
        }
      } catch (e: any) {
        setErr(String(e?.message || e));
      }
    })();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [department, queryDepartmentId]);

  useEffect(() => {
    if (!depId) {
      setLocations([]);
      setLocationId("");
      setEls([]);
      setElId("");
      return;
    }
    (async () => {
      try {
        const r = await fetch(`/api/locations?departmentId=${depId}`);
        const j = await r.json();
        if (j.ok) {
          setLocations(j.items || []);
          // Pre-fill locationId from query if provided
          if (queryLocationId && typeof queryLocationId === "string") {
            const foundLoc = j.items.find(
              (l: LocationItem) => l.id === queryLocationId
            );
            if (foundLoc) {
              setLocationId(foundLoc.id);
            } else {
              setLocationId("");
            }
          } else if (j.items?.every((l: LocationItem) => l.id !== locationId)) {
            setLocationId("");
          }
        } else {
          setErr(j.error || "Failed to load locations");
        }
      } catch (e: any) {
        setErr(String(e?.message || e));
      }
    })();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [depId, queryLocationId]);

  useEffect(() => {
    if (!locationId) {
      setEls([]);
      setElId("");
      return;
    }
    (async () => {
      try {
        const r = await fetch(`/api/elements?locationId=${locationId}`);
        const j = await r.json();
        if (j.ok) {
          setEls(j.items || []);
          // Pre-fill elementId from query if provided
          if (queryElementId && typeof queryElementId === "string") {
            const foundEl = j.items.find(
              (e: ElementItem) => e.id === queryElementId
            );
            if (foundEl) {
              setElId(foundEl.id);
            } else {
              setElId("");
            }
          } else if (j.items?.every((e: ElementItem) => e.id !== elId)) {
            setElId("");
          }
        } else {
          setErr(j.error || "Failed to load equipment");
        }
      } catch (e: any) {
        setErr(String(e?.message || e));
      }
    })();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [locationId]);

  const isLoggedIn = status === "authenticated";
  const isAnonymous = status === "unauthenticated";

  // Compress and convert image to base64
  const compressImage = async (file: File): Promise<string> => {
    return new Promise((resolve, reject) => {
      try {
        // If not an image, just return as-is
        if (!file.type.startsWith("image/")) {
          const reader = new FileReader();
          reader.onload = () => {
            if (typeof reader.result === "string") {
              resolve(reader.result);
            } else {
              reject(new Error("Failed to read file"));
            }
          };
          reader.onerror = reject;
          reader.readAsDataURL(file);
          return;
        }

        const reader = new FileReader();
        reader.onload = (e) => {
          const img = new Image();
          img.onload = () => {
            // Calculate new dimensions (max 1280px for smaller file size, maintaining aspect ratio)
            const maxDimension = 1280;
            let width = img.width;
            let height = img.height;

            if (width > maxDimension || height > maxDimension) {
              if (width > height) {
                height = (height / width) * maxDimension;
                width = maxDimension;
              } else {
                width = (width / height) * maxDimension;
                height = maxDimension;
              }
            }

            // Create canvas and compress
            const canvas = document.createElement("canvas");
            canvas.width = width;
            canvas.height = height;
            const ctx = canvas.getContext("2d");
            if (!ctx) {
              reject(new Error("Failed to get canvas context"));
              return;
            }

            // Draw and compress
            ctx.drawImage(img, 0, 0, width, height);

            // Convert to base64 with quality 0.7 (balanced quality and size)
            const compressedDataUrl = canvas.toDataURL("image/jpeg", 0.7);
            resolve(compressedDataUrl);
          };
          img.onerror = reject;
          img.src = e.target?.result as string;
        };
        reader.onerror = reject;
        reader.readAsDataURL(file);
      } catch (error) {
        reject(error);
      }
    });
  };

  const onSubmit = async (e: FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setResult(null);
    setErr(null);

    // Convert uploaded files to base64 with compression
    const fileAttachments: string[] = [];
    for (const file of uploadedFiles) {
      try {
        const base64 = await compressImage(file);
        fileAttachments.push(base64);
      } catch (error) {
        console.error("Error converting file to base64:", error);
      }
    }

    const body = {
      title,
      description,
      severity: Number(severity),
      departmentId: depId || undefined,
      locationId: locationId || undefined,
      elementId: elId || undefined,
      department: legacyDepartment || undefined,
      attachments: fileAttachments,
      ...(isAnonymous && { anonymousEmail, anonymousName }),
    };

    try {
      const resp = await fetch("/api/tickets/create", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(body),
      });
      const json = (await resp.json()) as CreateResp;
      setLoading(false);
      setResult(json);
      if (json.ok) {
        // open custom dialog instead of window.alert
        setDialogOpen(true);
      }
    } catch (e: any) {
      setLoading(false);
      setErr(String(e?.message || e));
    }
  };

  const copy = (text?: string) => {
    if (!text) return;
    navigator.clipboard?.writeText(text).catch(() => {});
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files) return;

    const newFiles = Array.from(files);
    setUploadedFiles((prev) => [...prev, ...newFiles]);
  };

  const removeFile = (index: number) => {
    setUploadedFiles((prev) => prev.filter((_, i) => i !== index));
  };

  const success = result && result.ok ? result : null;

  return (
    <main className='max-w-3xl mx-auto px-4 py-6'>
      {/* Header */}
      <header className='flex items-center gap-3'>
        <Button asChild variant='outline'>
          <Link href='/tickets'>
            <ArrowLeft className='w-4 h-4' />
            {t("common.back")}
          </Link>
        </Button>
        <h1 className='text-2xl font-bold ml-1'>{t("createTicket.title")}</h1>
        <div className='ml-auto flex items-center gap-3'>
          {isAnonymous && (
            <Button asChild variant='outline'>
              <Link href='/login'>{t("nav.signIn")}</Link>
            </Button>
          )}
        </div>
      </header>

      {/* Anonymous banner */}
      {isAnonymous && (
        <div className='alert alert-warning mt-4'>
          <AlertCircle />
          <div>
            <p className='font-semibold'>
              {t("createTicket.anonymous.banner")}
            </p>
            <p className='text-sm opacity-90'>
              {t("createTicket.anonymous.message")}
            </p>
          </div>
        </div>
      )}

      {/* Error banner */}
      {err && (
        <div className='alert alert-error mt-3'>
          <AlertCircle />
          <div>
            <p className='font-semibold'>Error</p>
            <p className='text-sm opacity-90'>{err}</p>
          </div>
        </div>
      )}

      {/* Logged-in badge */}
      {isLoggedIn && (
        <div className='alert alert-success mt-4'>
          <CheckCircle2 />
          <div>
            <p className='font-semibold'>
              {t("createTicket.loggedIn.banner")}{" "}
              {session?.user?.name || session?.user?.email}
            </p>
            <p className='text-sm opacity-90'>
              {t("createTicket.loggedIn.message")}
            </p>
          </div>
        </div>
      )}

      {/* Form */}
      <form onSubmit={onSubmit} className='card p-5 mt-4 space-y-4'>
        {/* Anonymous contact block */}
        {isAnonymous && (
          <>
            <div className='grid sm:grid-cols-2 gap-4'>
              <div className='space-y-2'>
                <Label htmlFor='anonymous-email'>
                  {t("createTicket.anonymous.email")} *
                </Label>
                <div className='relative'>
                  <Mail className='w-4 h-4 absolute left-3 top-1/2 -translate-y-1/2 opacity-60' />
                  <Input
                    id='anonymous-email'
                    type='email'
                    value={anonymousEmail}
                    onChange={(e) => setAnonymousEmail(e.target.value)}
                    required
                    placeholder={t("createTicket.anonymous.emailPlaceholder")}
                    className='pl-9'
                  />
                </div>
              </div>

              <div className='space-y-2'>
                <Label htmlFor='anonymous-name'>
                  {t("createTicket.anonymous.name")} *
                </Label>
                <div className='relative'>
                  <UserIcon className='w-4 h-4 absolute left-3 top-1/2 -translate-y-1/2 opacity-60' />
                  <Input
                    id='anonymous-name'
                    value={anonymousName}
                    onChange={(e) => setAnonymousName(e.target.value)}
                    required
                    placeholder={t("createTicket.anonymous.namePlaceholder")}
                    className='pl-9'
                  />
                </div>
              </div>
            </div>

            <hr
              className='border-t'
              style={{ borderColor: "var(--color-border)" }}
            />
          </>
        )}

        {/* Ticket basics */}
        <div className='space-y-2'>
          <Label htmlFor='title'>{t("createTicket.fields.title")}</Label>
          <Input
            id='title'
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            required
            placeholder='Short, clear summary'
          />
        </div>

        <div className='space-y-2'>
          <Label htmlFor='description'>
            {t("createTicket.fields.description")}
          </Label>
          <Textarea
            id='description'
            value={description}
            onChange={(e) => setDescription(e.target.value)}
            required
            rows={5}
            placeholder='Describe the issue and any relevant context'
          />
        </div>

        <div className='space-y-2'>
          <Label htmlFor='severity'>{t("createTicket.fields.severity")}</Label>
          <div className='flex flex-col gap-2'>
            <input
              id='severity'
              type='range'
              min={0}
              max={5}
              value={severity}
              onChange={(e) => setSeverity(parseInt(e.target.value || "0", 10))}
              className='w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider'
              style={{
                background: `linear-gradient(to right, #10b981 0%, #10b981 ${
                  (severity / 5) * 100
                }%, #e5e7eb ${(severity / 5) * 100}%, #e5e7eb 100%)`,
              }}
            />
            <div className='flex justify-between text-xs text-gray-600'>
              <span>0</span>
              <span className='font-semibold text-lg'>{severity}</span>
              <span>5</span>
            </div>
          </div>
        </div>

        {/* Department / Location / Equipment */}
        <div className='grid sm:grid-cols-3 gap-4'>
          <div className='space-y-2'>
            <Label htmlFor='department'>
              {t("createTicket.fields.department")}
              {department &&
                typeof department === "string" &&
                deps.find((d) => d.id === department) && (
                  <span className='ml-2 text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full'>
                    {t("createTicket.departmentPreselected")}
                  </span>
                )}
            </Label>
            <Select value={depId} onValueChange={setDepId} required>
              <SelectTrigger id='department'>
                <SelectValue
                  placeholder={
                    t("common.select") +
                    " " +
                    t("createTicket.fields.department").toLowerCase()
                  }
                />
              </SelectTrigger>
              <SelectContent>
                {deps.map((d) => (
                  <SelectItem key={d.id} value={d.id}>
                    {d.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className='space-y-2'>
            <Label htmlFor='location'>
              {t("createTicket.fields.location") || "Location"}
            </Label>
            <Select
              value={locationId || ""}
              onValueChange={setLocationId}
              disabled={!depId}
            >
              <SelectTrigger id='location'>
                <SelectValue
                  placeholder={
                    !depId
                      ? t("createTicket.selectDepartmentFirst") ||
                        "Select department first"
                      : t("common.select") +
                        " " +
                        (
                          t("createTicket.fields.location") || "location"
                        ).toLowerCase()
                  }
                />
              </SelectTrigger>
              <SelectContent>
                {locations.map((loc) => (
                  <SelectItem key={loc.id} value={loc.id}>
                    {loc.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className='space-y-2'>
            <Label htmlFor='element'>
              {t("createTicket.fields.equipment") ||
                t("createTicket.fields.element") ||
                "Equipment"}
            </Label>
            <Select
              value={elId || "none"}
              onValueChange={(value) => setElId(value === "none" ? "" : value)}
              disabled={!locationId}
            >
              <SelectTrigger id='element'>
                <SelectValue
                  placeholder={
                    !locationId
                      ? t("createTicket.selectLocationFirst") ||
                        "Select location first"
                      : t("common.select") +
                        " " +
                        (
                          t("createTicket.fields.equipment") ||
                          t("createTicket.fields.element") ||
                          "equipment"
                        ).toLowerCase()
                  }
                />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value='none'>(none)</SelectItem>
                {els.map((el) => (
                  <SelectItem key={el.id} value={el.id}>
                    {el.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* File Upload */}
        <div className='space-y-2'>
          <Label htmlFor='file-upload'>
            {t("createTicket.fields.attachment")}
          </Label>
          <div className='border-2 border-dashed border-input rounded-md p-4'>
            <input
              type='file'
              id='file-upload'
              multiple
              accept='image/*'
              onChange={handleFileChange}
              className='hidden'
            />
            <Label
              htmlFor='file-upload'
              className='flex flex-col items-center justify-center cursor-pointer hover:opacity-80 transition'
            >
              <Upload className='w-8 h-8 text-muted-foreground mb-2' />
              <span className='text-sm text-muted-foreground'>
                {t("createTicket.fileUpload.clickToUpload")}
              </span>
              <span className='text-xs text-muted-foreground mt-1'>
                {t("createTicket.fileUpload.autoCompress")}
              </span>
            </Label>
          </div>

          {/* File Preview */}
          {uploadedFiles.length > 0 && (
            <div className='grid grid-cols-2 sm:grid-cols-3 gap-3 mt-3'>
              {uploadedFiles.map((file, index) => (
                <div
                  key={index}
                  className='relative group border rounded-md overflow-hidden bg-card'
                >
                  {file.type.startsWith("image/") ? (
                    <img
                      src={URL.createObjectURL(file)}
                      alt={file.name}
                      className='w-full h-32 object-cover'
                    />
                  ) : (
                    <div className='w-full h-32 flex items-center justify-center bg-muted'>
                      <ImageIcon className='w-8 h-8 text-muted-foreground' />
                    </div>
                  )}
                  <button
                    type='button'
                    onClick={() => removeFile(index)}
                    className='absolute top-2 right-2 p-1.5 bg-red-500 hover:bg-red-600 active:bg-red-700 text-white rounded-full shadow-lg z-10'
                    title={
                      t("common.remove") +
                      " " +
                      t("createTicket.fields.attachment").toLowerCase()
                    }
                  >
                    <X className='w-4 h-4' />
                  </button>
                  <div className='absolute bottom-0 left-0 right-0 bg-black/70 text-white text-xs p-2 truncate'>
                    {file.name}
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>

        {/* Actions */}
        <div className='pt-2 flex items-center gap-3'>
          <Button type='submit' disabled={loading}>
            {loading ? (
              <>
                <div className='w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin' />
                {t("createTicket.buttons.creating")}
              </>
            ) : (
              <>
                <Save className='w-4 h-4' />
                {t("createTicket.buttons.create")}
              </>
            )}
          </Button>
          <Button asChild variant='outline'>
            <Link href='/tickets'>{t("common.cancel")}</Link>
          </Button>
        </div>
      </form>

      {/* Server result (error case already above) */}
      {result && !result.ok && (
        <section className='mt-4'>
          <div className='alert alert-error'>
            <AlertCircle />
            <div>
              <p className='font-semibold'>Error</p>
              <p className='text-sm opacity-90'>{result.error}</p>
            </div>
          </div>
        </section>
      )}

      {/* Success Dialog */}
      <Dialog
        open={dialogOpen && !!success}
        onClose={() => setDialogOpen(false)}
        title={t("createTicket.success.title")}
        footer={
          <>
            <Button
              variant='outline'
              onClick={() => {
                setDialogOpen(false);
                // Reset all form fields
                setTitle("");
                setDescription("");
                setSeverity(2);
                setUploadedFiles([]);
                setAnonymousEmail("");
                setAnonymousName("");
                setLocationId("");
                setElId("");
                setErr(null);
                setResult(null);
                // Reset department to first one if available
                if (deps.length > 0) {
                  setDepId(deps[0].id);
                }
              }}
              type='button'
            >
              {t("common.close")}
            </Button>
            {success?.ok && (
              <Button asChild>
                <Link href={`/tickets/${success.ticket.id}`}>
                  <ExternalLink className='w-4 h-4' />
                  {t("createTicket.buttons.viewTicket")}
                </Link>
              </Button>
            )}
          </>
        }
      >
        {success?.ok && (
          <div className='space-y-3'>
            <div className='flex items-center gap-2 text-[color:var(--color-success)]'>
              <CheckCircle2 className='w-5 h-5' />
              <span>{t("createTicket.success.message")}</span>
            </div>

            <p className='text-sm text-muted-foreground'>
              {t("createTicket.success.details")}
            </p>
          </div>
        )}
      </Dialog>
    </main>
  );
}
